#
# Copyright (c) Omnissa, LLC. All rights reserved.
# This product is protected by copyright and intellectual property laws in the
# United States and other countries as well as by international treaties.
# -- Omnissa Restricted
#
# This script executes uninstall tasks for file association of Omnissa
# Horizon Linux Client
#

import configparser
import fnmatch
import os
import pwd
import subprocess
import xml.etree.ElementTree as ET

sectionName = 'Added Associations'
patternAppsEntry = 'cn=*ou=*dc=*_*_horizon.desktop'
patternTypeFile = 'horizon-*.xml'

# Remove added association in mimeapps.list added by file association
def RemoveFAFromMimeAppsList(mimeAppsListPath, appEntry):
   config = configparser.ConfigParser()
   try:
      config.read(mimeAppsListPath)
   except configparser.Error:
      return

   for sectionName in config.sections():
      for optionName in config.options(sectionName):
         value = config.get(sectionName, optionName)
         appEntries = value.split(';')

         if appEntry in appEntries:
            appEntries.remove(appEntry)
            newValue = ';'.join(appEntries)
            if len(appEntries) > 1:
               config.set(sectionName, optionName, newValue)
            else:
               config.remove_option(sectionName, optionName)

   # Update the mimeapps.list
   with open(mimeAppsListPath, 'w') as mimeAppsListfile:
      config.write(mimeAppsListfile)

# Remove new types added by file association
def RemoveFATypes(mimePath, user):
   mimeTypePath = os.path.join(mimePath, 'packages')
   if not os.path.exists(mimeTypePath):
      return

   typeFiles = [f for f in os.listdir(mimeTypePath)
                if fnmatch.fnmatch(f, patternTypeFile)]

   for typeFile in typeFiles:
      typeFilePath = os.path.join(mimeTypePath, typeFile)
      os.remove(typeFilePath)

   # Update mime database for the user
   command = ["update-mime-database", mimePath]
   try:
      subprocess.run(["sudo", "-u", user] + command)
   except subprocess.CalledProcessError:
      pass

# Remove saved FA info in broker prefs
def RemoveFAPrefs(brokerPrefsPath):
   # Parse the broker prefs file
   try:
      tree = ET.parse(brokerPrefsPath)
   except ET.ParseError:
      return

   # Remove FileRedirection
   modified = False
   root = tree.getroot()
   for server in root:
      for faConfig in server.findall('FileRedirection'):
         server.remove(faConfig);
         modified = True

   if modified:
      tree.write(brokerPrefsPath, xml_declaration = True)

# Remove file association related configs
def RemoveFAConfig():
   userProfiles = pwd.getpwall()
   for profile in userProfiles:
      uid = profile.pw_uid
      user = profile.pw_name
      pwDir = profile.pw_dir
      # Only handle for normal user (uid >= 1000)
      if uid < 1000:
         continue
      appEntryDir = os.path.join(pwDir, '.local/share/applications')

      # Ignore user if no ~/.local/share/applications directory
      if not os.path.exists(appEntryDir):
         continue

      mimeAppsListPath = os.path.join(pwDir, '.config/mimeapps.list')
      mimeAppsListExist = os.path.exists(mimeAppsListPath)

      # Get list of app entry files for FA
      entryFiles = [f for f in os.listdir(appEntryDir)
                    if fnmatch.fnmatch(f, patternAppsEntry)]

      # Loop through the app entry files and remove them
      for entryFile in entryFiles:
         entryFilePath = os.path.join(appEntryDir, entryFile)
         os.remove(entryFilePath)
         if mimeAppsListExist:
            RemoveFAFromMimeAppsList(mimeAppsListPath, entryFile)

      # Remove new type created for FA
      mimePath = os.path.join(pwDir, '.local/share/mime')
      if os.path.exists(mimePath):
         RemoveFATypes(mimePath, user)

      # Remove FA config in broker prefs
      brokerPrefsPath = os.path.join(pwDir, '.omnissa/horizon-brokers-prefs')
      if os.path.exists(brokerPrefsPath):
         RemoveFAPrefs(brokerPrefsPath)

if __name__ == "__main__":
   RemoveFAConfig()
