#!/bin/bash
#
# Copyright (c) 2020 Omnissa, LLC. All rights reserved.
# This product is protected by copyright and intellectual property laws in the
# United States and other countries as well as by international treaties.
# -- Omnissa Public
#
# This script implements util functions for plugin manager on Linux.
#

# Global variables
BASHRC=".bashrc"
INSTALL_CHECK_SCRIPT="installation_check.py"
ENV_VAR_CHANGED=0


#----------------------------------------------------------------
#
# set_env_var --
#
#    Set environment variable to ~/.bashrc on Linux platform
#
#    Arguments:
#    1. name of environment variable
#    2. value of environment variable
#
#----------------------------------------------------------------

function set_env_var()
{
   local var=$1
   local loglevel=$2
   local bashrc="$(get_home_dir $username)/$BASHRC"

   ENV_VAR_CHANGED=1

   # update env variable
   export $var=$loglevel

   # Add or override env variable export in ~/.bashrc on Linux platform
   unset_env_var "$var"
   if [ -n "$loglevel" ]; then
      eval "echo 'export $var=$loglevel' >> $bashrc"
   fi

   log_debug "Set environment variable $var=$loglevel"
}


#----------------------------------------------------------------
#
# unset_env_var --
#
#    Delete environment variable export in ~/.bashrc
#
#    Arguments:
#    Name of environment variable
#
#----------------------------------------------------------------

function unset_env_var()
{
   local var=$1
   local bashrc="$(get_home_dir $username)/$BASHRC"

   ENV_VAR_CHANGED=1

   # Unset env variable
   unset $var

   # Remove env variable config in .bashrc
   eval sed -i -E "/^.*export[[:space:]]+$var=.*$/d" $bashrc

   log_debug "Unset environment variable $var"
}


#----------------------------------------------------------------
#
# read_loglevel_from_env --
#
#    Read loglevel config from environment variable
#
#    Arguments:
#    Name of environment variable
#
#----------------------------------------------------------------

function read_loglevel_from_env()
{
   local var=$(trim $1)
   local loglevel=${!var}

   echo $loglevel
}


#----------------------------------------------------------------
#
# format_date_string --
#
#    Format date string to epoch timestamp on Linux platform
#
#    Arguments:
#    Date string
#
#----------------------------------------------------------------

function format_date_string()
{
   local date="$1"
   local timestamp=$(date -d "$date" +%s)

   echo $timestamp
}


#----------------------------------------------------------------
#
# read_file_modified_time --
#
#    Read time when file data last modified on Linux platform
#
#    Arguments:
#    Name of file
#
#----------------------------------------------------------------

function read_file_modified_time()
{
   local file="$1"
   local file_time=$(stat -c %Y "$file")

   echo $file_time
}


#----------------------------------------------------------------
#
# check_installation_status --
#
#    Check installation status for a given plugin feature
#    Return 0 when feature is installed or status is unknown
#    Return 1 when feature is not installed
#
#    Arguments:
#    Feature short name
#
#----------------------------------------------------------------

function check_installation_status()
{
   local feature=$1
   local status
   local result=0

   # Check if python is supported, if not, optional installation
   # for features is not supported, all features are installed by default.
   if ! python3 --version &> /dev/null ; then
      return 0
   fi

   status=$(python3 $DCT_PATH/$INSTALL_CHECK_SCRIPT "$feature")

   # Only when query successfull and result is no, the feature is
   # determined to be not installed.
   if [ $? -eq 0 ] && [ $status == "no" ]; then
      result=1
   else
      result=0
   fi

   return $result
}


#----------------------------------------------------------------
#
# update_shell_context --
#
#    update shell context to make the log level change on
#    environment variable take effect in current shell on Linux
#    platform.
#
#----------------------------------------------------------------

function update_shell_context()
{
   # In sudo mode, do not execute shell, re-log in with sudo user
   # if DCT setting is for sudo user
   if [ $ENV_VAR_CHANGED -eq 1 ]; then
      if [ -n "$SUDO_USER" ]; then
         [ "$username" == "$SUDO_USER" ] && su - $username
      else
         [ "$username" == "$USER" ] && exec $SHELL -i
      fi
   fi
}
