#!/bin/bash
#
# Copyright (c) 2020 Omnissa, LLC. All rights reserved.
# This product is protected by copyright and intellectual property laws in the
# United States and other countries as well as by international treaties.
# -- Omnissa Public
#
# Utils of log collection for Linux Client.
#

# Define global variables for log collection on Linux Client
client_log_dir=""
client_log_glob="horizon-client-[0-9]*.log"
DEFAULT_DIR=""
DEFAULT_TARGET='horizon-log.tar.gz'
LOG_PACKAGE_SUFFIX=".tar.gz"

# Path aliases for DCT on Linux client
OMNISSA_TMP_DIR=""
PCOIP_LOG_DIR=""

#----------------------------------------------------------------
#
# check_directory --
#
#    Check if the directory to collect data exists
#
#----------------------------------------------------------------

function check_directory()
{
   if [ ! -n "${TMPDIR}" ] ; then
      export TMPDIR=/tmp
   fi

   OMNISSA_TMP_DIR="$TMPDIR/omnissa-$username"
   client_log_dir="$OMNISSA_TMP_DIR"
   # Find the directory that logs are stored.
   if [ ! -d "$client_log_dir" ] ; then
      throw_exception "The log directory $OMNISSA_TMP_DIR does not exist."
   fi

   # Get log path for PCoIP log. Sometimes, PCoIP log path will be truncated
   # if $TMPDIR/teradici-$username is too long.
   # So find the longest dir name that matches $TMPDIR/teradici-$username
   local pcoip_path="$TMPDIR/teradici-$username"
   for dir in $(ls -d "$TMPDIR"/* 2>/dev/null) ; do
      if [[ "$pcoip_path" =~ "$dir"* ]] && [ ${#dir} -gt ${#PCOIP_LOG_DIR} ]; then
         PCOIP_LOG_DIR="$dir"
      fi
   done
}


#----------------------------------------------------------------
#
# package_dir --
#
#    Package directory to a tarball
#
#    Arguments:
#    1. name fo tarball
#    2. directory to be packaged
#
#----------------------------------------------------------------

function package_dir()
{
   local package=$1
   local dir=$2

   tar czf "$package" --sort=name "$dir"
   return $?
}


#----------------------------------------------------------------
#
# cpLog --
#
#    Copy log file from directory. Only log files newer than dest file will be copied.
#
#    Note:
#    Check whether need to copy the PCoIP log and copy logs
#    Directory name may have space, and file name have the wildcard character.
#    So if use quotes to enclose the file name, ls command won't find it, and must
#    put the file name outside of the quotes. So 4 arguments are needed.
#
#    Arguments:
#    1. Source directory
#    2. File pattern (The files match the pattern will be copied)
#    3. Dest file (only files new than it will be copied)
#    4. Target diretory
#
#----------------------------------------------------------------

function cpLog()
{
   local cmpLogDir="$1"
   local cmpLogGlob="$2"
   local destViewLogFile="$3"
   local destTargetDir="$4"
   # In one connection, there are maybe more than one PCoIP/mks
   # logs and all should be collected. Use the most recently
   # view log's first line to get the view log's create time,
   # and then compare it with the mtime of
   # PCoIP/mks log, if PCoIP/mks is newer, just collect it.

   local destViewLogTime=$(date -d "$(head -n 1 "$destViewLogFile" | sed -e 's/: horizon-client.*$//')" "+%s")
   for logFile in $(ls -t "$cmpLogDir"/$cmpLogGlob 2>/dev/null) ; do
      local cmpLogTime=$(stat -c %Y "$logFile")
      if [ $cmpLogTime -lt $destViewLogTime ] ; then
         return
      fi
      if ! cp -p "$logFile" "$destTargetDir" ; then
         log_error "Unable to copy log file $logFile to $destTargetDir."
      fi
   done
}


#----------------------------------------------------------------
#
# cpThinPrintLog --
#
#    Legacy function: Copy thinprint log.
#
#    Arguments:
#    1. source dir
#    2. dest dir
#
#----------------------------------------------------------------

function cpThinPrintLog()
{
   local srcDir="$1"
   local destDir="$2"

   if [ -d "$srcDir" ]; then
      pushd "$srcDir" > /dev/null
      mkdir -p "$destDir/thnuclnt"
      find . -name "thnuclnt.log" -exec cp -p --parent {} "$destDir/thnuclnt" \;
      popd > /dev/null
   fi
}


#----------------------------------------------------------------
#
# cpScannerSerialLog --
#
#    Legacy function: Copy log for scanner serial
#
#    Arguments:
#    1. source dir
#    2. dest dir
#
#----------------------------------------------------------------

function cpScannerSerialLog()
{
   local srcDir="$1"
   local destDir="$2"

   if [ -d "$srcDir" ]; then
      pushd "$srcDir" > /dev/null
      mkdir -p "$destDir/scanner_serial"
      find . -name "ft*.log" -exec cp -p --parent {} "$destDir/scanner_serial" \;
      popd > /dev/null
   fi
}


#----------------------------------------------------------------
#
# collect_data_all --
#
#    Legacy function: Collect all logs/dumps for Linux Client
#
#----------------------------------------------------------------

function collect_data_all()
{
   #Ensure scanner_serial log collection under root
   if [ "root" == "$username" ] ; then
      touch $TMPDIR/omnissa-$username/horizon-client-0.log
   fi

   viewLogGlob="horizon-client-[0-9]*.log"
   # Ensure at least one log file exists.
   if ! ls "$TMPDIR/omnissa-$username"/$viewLogGlob &>/dev/null ; then
      throw_exception "No log found in $OMNISSA_TMP_DIR."
   fi

   fileToCp=$(ls -t "$OMNISSA_TMP_DIR"/$viewLogGlob | head -n 1)
   if [ -z "$fileToCp" ] ; then
      throw_exception "Unable to locate log file in $OMNISSA_TMP_DIR."
   fi

   if ! cp -p "$fileToCp" "$tmpdir/$targetDirectory" ; then
      throw_exception "Unable to copy log file $fileToCp to $tmpdir/$targetDirectory."
   fi

   viewLogFile=$fileToCp

   # Copy the usb logs
   usbLogGlob="horizon-view-usb*-[0-9]*.log"
   cpLog "$OMNISSA_TMP_DIR" "$usbLogGlob" "$viewLogFile" "$tmpdir/$targetDirectory"

   # Copy the mks logs
   mksLogGlob="*protocol-[0-9]*.log"
   cpLog "$OMNISSA_TMP_DIR" "$mksLogGlob" "$viewLogFile" "$tmpdir/$targetDirectory"

   hznMediaProviderLogDir="$OMNISSA_TMP_DIR/HZNMediaProvider"
   if [ -d "$hznMediaProviderLogDir" ] ; then
      if ! cp -p -r "$hznMediaProviderLogDir" "$tmpdir/$targetDirectory" ; then
         log_error "Unable to copy log file $hznMediaProviderLogDir to $tmpdir/$targetDirectory."
      fi
   fi

   # Copy the pcoip_client logs
   pcoipLogDir="$TMPDIR/teradici-$username"
   pcoipLogGlob="pcoip_client_*"
   cpLog "$pcoipLogDir" "$pcoipLogGlob" "$viewLogFile" "$tmpdir/$targetDirectory"

   # Copy the vdpServiceClient logs
   vdpServiceGlob="horizon-vdpService-Client-[0-9]*.log"
   cpLog "$OMNISSA_TMP_DIR" "$vdpServiceGlob" "$viewLogFile" "$tmpdir/$targetDirectory"

   # Copy the horizon-rdeSvc logs
   rdeSvcLogGlob="horizon-rdeSvc-[0-9]*.log"
   cpLog "$OMNISSA_TMP_DIR" "$rdeSvcLogGlob" "$viewLogFile" "$tmpdir/$targetDirectory"

   # Copy the horizon-MKSVchanClient logs
   MKSVchanClientLogGlob="horizon-MKSVchanClient-[0-9]*.log"
   cpLog "$OMNISSA_TMP_DIR" "$MKSVchanClientLogGlob" "$viewLogFile" "$tmpdir/$targetDirectory"

   # Copy the horizonRdpvcBridge logs
   OldRdpVCBridgeLogGlob="VChan-Client.log"
   cpLog "$OMNISSA_TMP_DIR" "$OldRdpVCBridgeLogGlob" "$viewLogFile" "$tmpdir/$targetDirectory"
   RdpVCBridgeLogGlob="horizon-rdpvcbridge-Client-[0-9]*.log"
   cpLog "$OMNISSA_TMP_DIR" "$RdpVCBridgeLogGlob" "$viewLogFile" "$tmpdir/$targetDirectory"

   # Copy the horizon-RTAV logs
   RTAVLogGlob="horizon-RTAV-[0-9]*.log"
   cpLog "$OMNISSA_TMP_DIR" "$RTAVLogGlob" "$viewLogFile" "$tmpdir/$targetDirectory"

   # Copy the horizon-blastproxy logs
   BlastProxyLogGlob="horizon-blastproxy-[0-9]*.log*"
   cpLog "$OMNISSA_TMP_DIR" "$BlastProxyLogGlob" "$viewLogFile" "$tmpdir/$targetDirectory"

   # Copy the tsdr logs
   TsdrLogGlob="horizon-ViewTsdr-Client-[0-9]*.log"
   cpLog "$OMNISSA_TMP_DIR" "$TsdrLogGlob" "$viewLogFile" "$tmpdir/$targetDirectory"

   # Copy the tsmmr logs
   TsmmrLogGlob="horizon-ViewTsmmr-Client-[0-9]*.log"
   cpLog "$OMNISSA_TMP_DIR" "$TsmmrLogGlob" "$viewLogFile" "$tmpdir/$targetDirectory"

    # Copy the crtbora logs
   CrtboraLogGlob="horizon-crtbora-[0-9]*.log"
   cpLog "$OMNISSA_TMP_DIR" "$CrtboraLogGlob" "$viewLogFile" "$tmpdir/$targetDirectory"

   # Copy the usbredirectionclient logs
   UsbRedirectionClientLogGlob="horizon-UsbRedirectionClient-[0-9]*.log"
   cpLog "$OMNISSA_TMP_DIR" "$UsbRedirectionClientLogGlob" "$viewLogFile" "$tmpdir/$targetDirectory"

   # Copy the html5mmr logs
   Html5mmrLogGlob="horizon-html5Client-[0-9]*.log"
   cpLog "$OMNISSA_TMP_DIR" "$Html5mmrLogGlob" "$viewLogFile" "$tmpdir/$targetDirectory"

   # Copy the logs of videoplayer when html5mmr is enabled
   VideoplayerLogGlob="horizon-videoplayer-[0-9]*.log"
   cpLog "$OMNISSA_TMP_DIR" "$VideoplayerLogGlob" "$viewLogFile" "$tmpdir/$targetDirectory"

   # Copy the logs of integrated print
   PrintClientLogGlob="horizon-print-client-[0-9]*.log"
   cpLog "$OMNISSA_TMP_DIR" "$PrintClientLogGlob" "$viewLogFile" "$tmpdir/$targetDirectory"
   PrintVdpPluginLogGlob="horizon-print-vdpplugin-[0-9]*.log"
   cpLog "$OMNISSA_TMP_DIR" "$PrintVdpPluginLogGlob" "$viewLogFile" "$tmpdir/$targetDirectory"

   # Copy the thin print logs
   cpThinPrintLog "$HOME/.thnuclnt" "$tmpdir/$targetDirectory"

   # Copy the logs of Scanner&Serial
   cpScannerSerialLog "$OMNISSA_TMP_DIR" "$tmpdir/$targetDirectory"

   # Copy logs for DCT script
   dctLogGlob="horizon-dct-*.log"
   cpLog "$OMNISSA_TMP_DIR" "$dctLogGlob" "$viewLogFile" "$tmpdir/$targetDirectory"

   # Collect data by plugin feature/components
   for file in ${PLUGIN_LIST[@]}; do
      collect_data_by_plugin_config "$file" "$dctLogTimeFilter"
   done
}


#----------------------------------------------------------------
#
# get_dct_log_dir --
#
#    Get the directory to store log file for DCT script.
#
#----------------------------------------------------------------

function get_dct_log_dir()
{
   local log_dir
   local user=$(get_current_user)

   if [ ! -n "${TMPDIR}" ] ; then
      export TMPDIR=/tmp
   fi

   log_dir="$TMPDIR/omnissa-$user"

   echo $log_dir
}


#----------------------------------------------------------------
#
# get_expanded_temp_log_dir --
#
#    Expand path includes path alias OMNISSA_TMP_DIR.
#    On Linux client, the OMNISSA_TMP_DIR may be pointed to multiple
#    directories: omnissa-$user or omnissa-$user-<suffix> ...
#
#    Arguments:
#    source dir to be expanded
#
#----------------------------------------------------------------

function get_expanded_temp_log_dir()
{
   local srcdir="$1"

   declare -a log_dirs

   declare -a tmp_dirs
   tmp_dirs+=($OMNISSA_TMP_DIR)
   local other_tmp_dir=`ls -d "$OMNISSA_TMP_DIR"-* 2>/dev/null`
   tmp_dirs+=($other_tmp_dir)

   for dir in ${tmp_dirs[@]}; do
      local expanded_dir=${srcdir//OMNISSA_TMP_DIR/$dir}
      log_dirs+=($expanded_dir)
   done

   echo "${log_dirs[@]}"
}
