#
# Copyright (c) Omnissa, LLC. All rights reserved.
# This product is protected by copyright and intellectual property laws in the
# United States and other countries as well as by international treaties.
# -- Omnissa Restricted
#
# This script executes install/uninstall tasks for Horizon URL redirection
# of Linux Client
#

import getopt
import json
import os
import os.path
import pwd
import sys

helpInfo = 'Usage: python_script.py [-i|-u]'
errorInvalidInput = 'Invalid arguments. ' + helpInfo

chromeExtensionPath = '.config/google-chrome/Default/Extensions/lfidjngibpklhhijphdmbmedchiiolgk'
chromeUrlAllowlistPath = '/etc/opt/chrome/policies/managed'
edgeExtensionPath = '.config/microsoft-edge/Default/Extensions/lfidjngibpklhhijphdmbmedchiiolgk'
edgeUrlAllowlistPath = '/etc/opt/edge/policies/managed'
firefoxConfigPath = '.mozilla/managed-storage'
firefoxSnapConfigPath = 'snap/firefox/common/.mozilla/managed-storage'
firefoxHandlerFile = "handlers.json"
firefoxManagedStorage = '/usr/lib/mozilla/managed-storage'
firefoxManagedStorage64 = '/usr/lib64/mozilla/managed-storage'
urlAllowlistFile = 'horizon-urlAllowlist.json'
urlConfigFile = '/etc/omnissa/broker-url-config'
urlExtensionChromium = 'data.json'
urlExtensionFirefox = 'url_redirection@omnissa.com.json'
urlFilterEntry = '/usr/share/applications/urlfilter.desktop'
chromeExtensionHostManifestPath = '/.config/google-chrome/NativeMessagingHosts/horizon_url_native_host.json'

# Config browser for URL redirection
def ConfigBrowser():
   userProfiles = pwd.getpwall()
   for profile in userProfiles:
      uid = profile.pw_uid
      pwDir = profile.pw_dir
      # Only handle for normal user (uid > 1000)
      if uid < 1000:
         continue
      firefoxConfigDir = os.path.join(pwDir, '.mozilla/firefox')
      if os.path.exists(firefoxConfigDir):
         SetFirefoxHandlers(firefoxConfigDir)

# Set Firefox handler for horizon-client:// and vmware-view://
def SetFirefoxHandlers(firefoxConfigDir):
   searchpath = os.path.expanduser(firefoxConfigDir)
   for root, dirs, files in os.walk(searchpath):
      result = []
      if firefoxHandlerFile in files:
         result.append(os.path.join(root, firefoxHandlerFile))

      if not len(result) > 0:
         continue

      handlerFilePath = result[0]
      try:
         with open(handlerFilePath) as fin:
            data = json.load(fin)

         # Set Firefox to use system default handler for horizon-client:// and vmware-view://
         schemes = data.get('schemes')
         if schemes is not None:
            client_config = dict()
            # Set handler action as use system default (By Firefox spec: useSystemDefault = 4)
            client_config['action']=4
            schemes['horizon-client']=client_config
            schemes['vmware-view']=client_config
            with open(handlerFilePath,'w') as fout:
               json.dump(data, fout)
            break
      except ValueError:
         continue

# Create url Allowlist config files for chromium browser
def CreateChromiumUrlAllowlist(urlAllowlistPath):
   if not os.path.exists(urlAllowlistPath):
      os.makedirs(urlAllowlistPath, 0o755);

   chromiumUrlAllowlistConfig = os.path.join(urlAllowlistPath,
                                             urlAllowlistFile)
   urlAllowList = {"URLAllowlist": ["horizon-client:*", "vmware-view:*"]}
   with open(chromiumUrlAllowlistConfig, "w+") as f:
      json.dump(urlAllowList, f)

   os.chmod(chromiumUrlAllowlistConfig, 0o744)


# Remove url Allowlist config files for chromium browser
def RemoveChromiumUrlAllowlist(urlAllowlistPath):
   chromiumUrlAllowlistConfig = os.path.join(urlAllowlistPath,
                                             urlAllowlistFile)
   if os.path.exists(chromiumUrlAllowlistConfig):
      os.remove(chromiumUrlAllowlistConfig)

# Remove url config files for chromium browser
def RemoveChromiumUrlConfig(pwDir, extensionPath):
   configPath = os.path.join(pwDir,
                             extensionPath)
   if os.path.exists(configPath):
      for version in os.listdir(configPath):
         chromiumConfig = os.path.join(configPath,
                                       version,
                                       urlExtensionChromium)
         if os.path.exists(chromiumConfig):
            os.remove(chromiumConfig)

# Remove url config files for filter app and firefox/chrome/edge extension
def RemoveUrlConfig():
   if os.path.exists(urlConfigFile):
      os.remove(urlConfigFile);
   if os.path.exists(urlFilterEntry):
      os.remove(urlFilterEntry);

   # Remove chromium url Allowlist config
   RemoveChromiumUrlAllowlist(chromeUrlAllowlistPath)
   RemoveChromiumUrlAllowlist(edgeUrlAllowlistPath)

   # Remove firefox url config
   firefoxExtensionConfig = os.path.join(firefoxManagedStorage,
                                         urlExtensionFirefox)
   firefoxExtensionConfig64 = os.path.join(firefoxManagedStorage64,
                                         urlExtensionFirefox)
   if os.path.exists(firefoxExtensionConfig):
      os.remove(firefoxExtensionConfig)
   if os.path.exists(firefoxExtensionConfig64):
      os.remove(firefoxExtensionConfig64)

   # Remove url config for each user
   userProfiles = pwd.getpwall()
   for profile in userProfiles:
      uid = profile.pw_uid
      pwDir = profile.pw_dir
      # Only handle for normal user (uid >= 1000)
      if uid < 1000:
         continue

      # Remove firefox url config file
      firefoxConfig = os.path.join(pwDir,
                                   firefoxConfigPath,
                                   urlExtensionFirefox)
      if os.path.exists(firefoxConfig):
         os.remove(firefoxConfig)

      # Remove firefox(snap installed) url config file
      firefoxSnapConfig = os.path.join(pwDir,
                                       firefoxSnapConfigPath,
                                       urlExtensionFirefox)
      if os.path.exists(firefoxSnapConfig):
         os.remove(firefoxSnapConfig)

      # Remove chrome url config files
      RemoveChromiumUrlConfig(pwDir, chromeExtensionPath)

      # Remove edge url config files
      RemoveChromiumUrlConfig(pwDir, edgeExtensionPath)
      
      #Remove Chrome Extension NMH host manifest file if present
      chromeExtHostFile = pwDir + chromeExtensionHostManifestPath
      if os.path.exists(chromeExtHostFile):
         os.remove(chromeExtHostFile)

# Install for URL redirection
def install():
   ConfigBrowser()
   CreateChromiumUrlAllowlist(chromeUrlAllowlistPath)
   CreateChromiumUrlAllowlist(edgeUrlAllowlistPath)

# Install for URL redirection
def uninstall():
   RemoveUrlConfig()

# Main function for script
def main(argv):
   try:
      opts, args = getopt.getopt(argv, "iuh")
   except getopt.GetoptError:
      print(errorInvalidInput)
      sys.exit(1)

   if len(opts) != 1:
      print(errorInvalidInput)
      sys.exit(1)

   option = opts[0][0]

   if option == '-i':
      install()
   elif option == '-u':
      uninstall()
   elif option == '-h':
      print(helpInfo)

if __name__ == "__main__":
   main(sys.argv[1:])
